/*************************************************************************
 * Spulenwickler
 *
 * Arduino Nano, 128x64 I2C OLED display
 *
 * Reinhard Zickwolff, 2023
 ************************************************************************/

#define PULS_IN   2   // Signal von Gabellichtschranke
#define PWM_OUT   5   // Motorsteuerung (FET Gate)
#define OSZI_PIN  6   // fuer Zeitmessung & Diagnose
#define BTN_DOWN  7   // Taster fuer Sollwert Eingabe
#define BTN_UP    8   //    "              "
#define BTN_ENTER 9   //    "              "

#define RPM_POTI A3      // Poti fuer Drehzahlsteuerung
#define MAX_VALUE 10000  // max. Windungsanzahl

//--------------------- OLED ---------------------------------
// OLED libraries
#include "SSD1306Ascii.h"
#include "SSD1306AsciiAvrI2c.h"

// OLED address 0x3C
#define I2C_ADDRESS 0x3C
// Define proper OLED RST_PIN if required.
#define RST_PIN -1

SSD1306AsciiAvrI2c oled;

//- - - - - - - - - Globale Variablen - - - - - - - - - - - - - - - - - - - - -
volatile int revCnt = 0;                // Windungszaehler
volatile bool windgComplete = false;    // Fertig Status
volatile unsigned long lastIntTime;     // fuer Entstoerung des Lichtschranken Signals
bool motorOn = false;                   // Motor Status
int pwmValue;                           // fuer Motor Drehzahl Steuerung
int maxCount = 0;                       // Soll Windungszahl

//-------------------------------------------------------------------------------
// PWM Wert auf OLED anzeigen
void showPWM(int val){
  oled.setCursor(1,1); oled.print("PWM :     ");
  oled.setCursor(60,1); oled.print(val);
}

//-------------------------------------------------------------------------------
// Sollwert fuer Spulenwindungen auf OLED anzeigen
void showMaxWndg(){
  oled.setCursor(1,3); oled.print("Stop:      ");
  oled.setCursor(60,3); oled.print(maxCount);
}

//-------------------------------------------------------------------------------
// Aktuelle Windungszahl anzeigen
void showCount(int count){
  oled.setCursor(1,5); oled.print("Wndg:     ");
  oled.setCursor(60,5); oled.print(count);
}

//-------------------------------------------------------------------------------
// PWM Wert ueber Poti einlesen
void setPWMvalue() {
  pwmValue = analogRead(RPM_POTI);               // 0 <= pwmValue <= 1023
  if (pwmValue < 20) pwmValue = 0;               // stop Motor
  if (pwmValue > 930) pwmValue = 4 * 255;        // max speed
  showPWM(pwmValue/4);
}

//-------------------------------------------------------------------------------
// Soll Windungszahl ueber Up/Down Taster eingeben
void setMaxCount(){ 
  bool buttonPressed = false;
  
  if (!digitalRead(BTN_UP)) {
    buttonPressed = true;
    maxCount++;
    if (maxCount > MAX_VALUE) maxCount = MAX_VALUE;
    delay(100);  // entprellen
  }
  else if (!digitalRead(BTN_DOWN)) {
    buttonPressed = true;
    maxCount--;
    if (maxCount < 0) maxCount = 0;
    delay(100);  // entprellen
  }
  if (buttonPressed) {
    revCnt = 0;
    showCount(revCnt);
    showMaxWndg();
  }  
}

//-------------------------------------------------------------------------------
// Interrupt Service Routine fuer Lichtschranken Impulse (neg. Flanke)
void countISR() {
  
  // Signal entprellen/entstoeren
  if ((millis() - lastIntTime) > 150) {
    digitalWrite(OSZI_PIN, HIGH);  // fuer Diagnose
    revCnt++;
    lastIntTime = millis();
    digitalWrite(OSZI_PIN, LOW);   // fuer Diagnose
  }  
  
  if (revCnt >= maxCount) {
    windgComplete = true;          // Spule ist fertig gewickelt
  }
    
}


//===============================================================================
void setup() {

  // Ports initialisieren
  pinMode(PULS_IN,INPUT_PULLUP);
  pinMode(BTN_DOWN,INPUT_PULLUP);
  pinMode(BTN_UP,INPUT_PULLUP);
  pinMode(BTN_ENTER,INPUT_PULLUP);
  pinMode(PWM_OUT,OUTPUT);
  
  pinMode(6,OUTPUT);  // Oszi test pin
  
  // OLED initialisieren
  oled.begin(&Adafruit128x64, I2C_ADDRESS);
  oled.setFont(fixed_bold10x15);
  oled.set1X();
  oled.clear();
  
  showMaxWndg();
  showCount(revCnt);
  
  lastIntTime = millis();
  attachInterrupt(digitalPinToInterrupt(PULS_IN), countISR, FALLING);
}

//===================== Hauptprogramm Schleife ==================================
void loop() {
  
  setPWMvalue();   
  setMaxCount();
  
  if (!digitalRead(BTN_ENTER)) { 
    revCnt = 0;
    showCount(revCnt);
    motorOn = true;
    windgComplete = false;
    delay(500); // entprellen
  
    while (!windgComplete && motorOn) {
      // Spule wickeln ...
      // windgComplete wird in Interrupt Routine gesetzt
      setPWMvalue();                  // Drehzahl Einstellung waehrend Wickeln ermoeglichen
      analogWrite(PWM_OUT, pwmValue/4);
      showCount(revCnt);
    }  
  
    if (windgComplete) {
      analogWrite(PWM_OUT, 0);        // stop motor
      motorOn = false;
      windgComplete = false;
      showCount(revCnt);
    } 
  } 
}



